<?php
session_start();

// 🔐 Secure access (optional): ensure only admins can access
if (!isset($_SESSION['admin_logged_in'])) {
    die("Unauthorized access.");
}

// File to store allowed IPs
$ip_file = __DIR__ . '/allowed_ips.txt';
$current_ip = $_SERVER['REMOTE_ADDR'];
$message = '';

// ✅ Add IP
if (isset($_POST['new_ip'])) {
    $new_ip = trim($_POST['new_ip']);
    if (filter_var($new_ip, FILTER_VALIDATE_IP)) {
        $ips = file($ip_file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        if (!in_array($new_ip, $ips)) {
            file_put_contents($ip_file, $new_ip . PHP_EOL, FILE_APPEND);
            $message = "✅ IP $new_ip added successfully.";
        } else {
            $message = "⚠️ IP $new_ip already exists.";
        }
    } else {
        $message = "❌ Invalid IP format.";
    }
}

// ❌ Delete IP
if (isset($_GET['delete'])) {
    $ip_to_delete = $_GET['delete'];
    $ips = file($ip_file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    $updated_ips = array_filter($ips, fn($ip) => $ip !== $ip_to_delete);
    file_put_contents($ip_file, implode(PHP_EOL, $updated_ips) . PHP_EOL);
    $message = "🗑️ IP $ip_to_delete removed.";
}

// Load allowed IPs
$allowed_ips = file($ip_file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
?>

<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>SMM Boost Rental - IP Manager</title>
  <style>
    body {
      font-family: Arial, sans-serif;
      background: #f7f9fb;
      margin: 0;
      padding: 0;
    }
    .container {
      max-width: 720px;
      margin: 40px auto;
      background: #fff;
      padding: 30px;
      border-radius: 12px;
      box-shadow: 0 0 12px rgba(0,0,0,0.1);
    }
    h1 {
      color: #2c3e50;
      margin-bottom: 20px;
    }
    .msg {
      padding: 10px;
      background: #e8f0fe;
      border-left: 4px solid #4285f4;
      margin-bottom: 20px;
      border-radius: 6px;
    }
    .form-group {
      display: flex;
      margin-bottom: 20px;
    }
    input[type="text"] {
      flex: 1;
      padding: 10px;
      border: 1px solid #ccc;
      border-radius: 6px;
      font-size: 16px;
    }
    button {
      background: #2d89ef;
      color: white;
      padding: 10px 18px;
      border: none;
      border-radius: 6px;
      margin-left: 10px;
      font-size: 16px;
      cursor: pointer;
    }
    button:hover {
      background: #1b66c9;
    }
    ul {
      list-style: none;
      padding-left: 0;
    }
    li {
      background: #f2f2f2;
      margin: 8px 0;
      padding: 10px 14px;
      border-radius: 6px;
      display: flex;
      justify-content: space-between;
      align-items: center;
    }
    .remove-link {
      color: red;
      text-decoration: none;
      font-size: 14px;
    }
    .current-ip {
      font-size: 15px;
      color: #555;
      margin-top: -10px;
      margin-bottom: 20px;
    }
  </style>
</head>
<body>
  <div class="container">
    <h1>SMM Boost Rental – Admin IP Manager</h1>
    <div class="current-ip">
      🌐 Your current IP: <strong><?= htmlspecialchars($current_ip) ?></strong>
    </div>

    <?php if ($message): ?>
      <div class="msg"><?= htmlspecialchars($message) ?></div>
    <?php endif; ?>

    <form method="POST">
      <div class="form-group">
        <input type="text" name="new_ip" placeholder="Enter IP address to allow" required>
        <button type="submit">Add IP</button>
      </div>
    </form>

    <h3>📜 Allowed IP Addresses</h3>
    <ul>
      <?php foreach ($allowed_ips as $ip): ?>
        <li>
          <?= htmlspecialchars($ip) ?>
          <a class="remove-link" href="?delete=<?= urlencode($ip) ?>" onclick="return confirm('Are you sure you want to remove this IP?')">Remove</a>
        </li>
      <?php endforeach; ?>
    </ul>
  </div>
</body>
</html>
